import { state, dom, NETPLAY_CONSOLES, DISC_BASED_CONSOLES, DOWNLOAD_TIMEOUT_MS } from './state.js';
import * as api from './api.js';

export function showNetplayAlert() {
    if (state.netplayAlertTimeoutId) {
        clearTimeout(state.netplayAlertTimeoutId);
    }

    if (dom.netplayTitle && dom.netplaySubtitle && dom.netplayStatusContainer) {
        dom.netplayTitle.textContent = "Online Games Available!";
        dom.netplaySubtitle.textContent = "Press Back/Select to check Lobby";
        dom.netplayTitle.classList.add('glow');
        dom.netplayStatusContainer.classList.add('visible');

        state.netplayAlertTimeoutId = setTimeout(hideNetplayAlert, 300000); // 5 minutes
    }
}

export function hideNetplayAlert() {
    if (state.netplayAlertTimeoutId) {
        clearTimeout(state.netplayAlertTimeoutId);
        state.netplayAlertTimeoutId = null;
    }

    if (dom.netplayTitle && dom.netplaySubtitle && dom.netplayStatusContainer) {
        dom.netplayStatusContainer.classList.remove('visible');
        dom.netplayTitle.classList.remove('glow');

        // Clear text after the fade-out transition completes
        setTimeout(() => {
            dom.netplayTitle.textContent = "";
            dom.netplaySubtitle.textContent = "";
        }, 500); // Must match the transition duration in CSS
    }
}

export function renderGameList(gamesToRender) {
    dom.gameListUl.innerHTML = '';
    if (!gamesToRender || gamesToRender.length === 0) {
        let message = '<li>No games found</li>';
        if (state.isShowingFavorites) message = '<li>No favorite games found for this console</li>';
        else if (state.isShowingDownloaded) message = '<li>No downloaded games found for this console</li>';
        else if (dom.gameSearchInput.value) message = '<li>No games found matching search</li>';
        dom.gameListUl.innerHTML = message;
        return;
    }

    gamesToRender.forEach((gameGroup, index) => {
        const li = document.createElement('li');
        
        let gameText = gameGroup.baseName;
        if (state.selectedConsoleName.startsWith('Arcade') && state.arcadeTitles && state.arcadeTitles[gameGroup.baseName]) {
            gameText = state.arcadeTitles[gameGroup.baseName];
        }

        if (state.favoriteGames[state.selectedConsoleName]?.includes(gameGroup.baseName)) {
            gameText += ' ⭐';
        }
        const gameTextNode = document.createTextNode(gameText);
        li.appendChild(gameTextNode);
        li.gameTextNode = gameTextNode;
        li.dataset.gameBaseNameForImage = gameGroup.baseName;

        if (gameGroup.versions.length > 1) {
            const arrowSpan = document.createElement('span');
            arrowSpan.classList.add('version-arrow');
            arrowSpan.textContent = '≡';
            li.appendChild(arrowSpan);
            li.dataset.hasVersions = 'true';
        } else {
            li.dataset.hasVersions = 'false';
        }

        li.addEventListener('click', (event) => {
            if (event.target !== li.querySelector('.version-arrow')) {
                selectGame(index, true, false);
                state.navigationMode = 'gameList';
            }
        });

        li.querySelector('.version-arrow')?.addEventListener('click', (event) => {
            event.stopPropagation();
            if (state.isPlayDownloadMenuActive || state.isSaveManageMenuActive || state.isSavePromptActive) return;
            openVersionMenu(gameGroup);
        });

        li.addEventListener('dblclick', () => handleGameSelectionAction(gameGroup));
        dom.gameListUl.appendChild(li);
    });
}

export async function selectGame(index, shouldHighlight = true, shouldScroll = true) {
    if (index < 0 || index >= state.currentGames.length) {
        if (state.selectedIndex !== -1 && state.currentGames.length > 0 && state.selectedIndex < state.currentGames.length) {
            index = state.selectedIndex;
        } else if (state.currentGames.length > 0) {
            index = 0;
        } else {
            index = -1;
        }
    }

    if (state.selectedIndex !== index || !dom.gameListUl.children[index]?.classList.contains('selected')) {
        if (state.selectedIndex !== -1 && dom.gameListUl.children[state.selectedIndex]) {
            dom.gameListUl.children[state.selectedIndex].classList.remove('selected');
        }
        state.selectedIndex = index;
        if (state.selectedIndex !== -1 && dom.gameListUl.children[state.selectedIndex]) {
            const selectedLi = dom.gameListUl.children[state.selectedIndex];
            selectedLi.classList.add('selected');
            if (shouldScroll) {
                selectedLi.scrollIntoView({ block: 'center', behavior: 'auto' });
            }
        }
    }

    if (shouldHighlight && state.selectedIndex !== -1 && state.currentGames[state.selectedIndex]) {
        highlightLetterForGame(state.currentGames[state.selectedIndex].baseName);
    }
    await updateGameImage(state.selectedIndex);
}

export function highlightLetterForGame(gameBaseName) {
    if (!state.shouldHighlightLetterButton || !gameBaseName) return;

    let titleToSort = gameBaseName;
    if (state.selectedConsoleName.startsWith('Arcade') && state.arcadeTitles && state.arcadeTitles[gameBaseName]) {
        titleToSort = state.arcadeTitles[gameBaseName];
    }
    
    let firstLetter = titleToSort.toUpperCase().charAt(0);
    if (!firstLetter.match(/[A-Z]/)) {
        firstLetter = '#';
    }
    const foundButton = Array.from(dom.letterButtons).find(b => b.dataset.letter === firstLetter);
    setActiveLetterButton(foundButton || null);
}

export async function updateGameImage(currentIndex) {
    let imageToShow = state.paths.defaultImageUrl;
    let altText = 'Game Screenshot';
    const gameAvailable = currentIndex !== -1 && state.currentGames && state.currentGames.length > currentIndex && state.currentGames[currentIndex];

    if (gameAvailable) {
        const selectedGameGroupImg = state.currentGames[currentIndex];
        const platformName = state.selectedConsoleName;
        const gameBaseNameForImage = dom.gameListUl.children[currentIndex]?.dataset?.gameBaseNameForImage || selectedGameGroupImg.baseName;

        if (platformName && gameBaseNameForImage) {
            const imageDir = `${state.paths.imagesBasePath}/${platformName}/Screenshot - Gameplay`.replace(/\\/g, '/');
            try {
                const screenshotFullPath = await window.electronAPI.findGameImage({ imageDir, gameTitle: gameBaseNameForImage });
                imageToShow = screenshotFullPath || state.paths.defaultImageUrl;
            } catch (error) {
                imageToShow = state.paths.defaultImageUrl;
            }
        }
    } else if (!state.currentGames || state.currentGames.length === 0) {
        imageToShow = state.paths.welcomeImageUrl;
        altText = 'Welcome';
    }

    dom.gameScreenshotImg.src = imageToShow;
    dom.gameScreenshotImg.alt = altText;
}

export function updateFavoriteStatusInList(updatedGameBaseName) {
    const listItems = dom.gameListUl.children;
    for (const listItem of listItems) {
        if (listItem.dataset.gameBaseNameForImage === updatedGameBaseName) {
            if (listItem.gameTextNode) {
                let gameText = updatedGameBaseName;
                if (state.selectedConsoleName.startsWith('Arcade') && state.arcadeTitles && state.arcadeTitles[updatedGameBaseName]) {
                    gameText = state.arcadeTitles[updatedGameBaseName];
                }

                if (state.favoriteGames[state.selectedConsoleName]?.includes(updatedGameBaseName)) {
                    gameText += ' ⭐';
                }
                listItem.gameTextNode.textContent = gameText;
            }
            break;
        }
    }
}

export function handleGameSelectionAction(gameGroup) {
    if (!gameGroup || state.isEmuAppRunning || state.isSupportPopupActive || state.isPlayDownloadMenuActive || state.isSaveManageMenuActive || state.isSavePromptActive || state.isLobbyMenuActive || state.isVirtualKeyboardActive) return;
    if (gameGroup.versions.length > 1) {
        openVersionMenu(gameGroup);
    } else if (gameGroup.versions.length === 1) {
        showPlayDownloadMenu(gameGroup);
    }
}

export function handleNewGameList(gameGroups) {
    dom.gameListUl.parentElement.scrollTop = 0;
    dom.letterButtons.forEach(button => button.classList.remove('active'));
    state.activeLetterButton = null;
    dom.letterJumpColumn.classList.remove('column-focused');
    state.shouldHighlightLetterButton = false;
    state.gameItems = gameGroups;
    state.currentGames = gameGroups;
    renderGameList(gameGroups);
    if (gameGroups && gameGroups.length > 0) {
        selectGame(0, true, false);
        if (state.navigationMode === 'initialConsoleList') {
            state.navigationMode = 'gameList';
        }
    } else {
        state.selectedIndex = -1;
        if (state.navigationMode === 'initialConsoleList') {
            state.navigationMode = 'gameList';
        }
        updateGameImage(-1);
    }
    state.shouldHighlightLetterButton = true;
    if (!state.isInitialSelectionActive) {
        const focusTarget = dom.gameListUl.children[state.selectedIndex] || (dom.gameListUl.children.length > 0 ? dom.gameListUl.children[0] : null);
        focusTarget?.focus();
    }
}

export function setActiveLetterButton(button) {
    if (state.activeLetterButton && state.activeLetterButton !== button) {
        state.activeLetterButton.classList.remove('active');
    }
    state.activeLetterButton = button;
    if (button) {
        button.classList.add('active');
    }
}

export function renderConsoleLists(consoleFolders) {
    dom.consoleListUl.innerHTML = '';
    dom.initialConsoleListUl.innerHTML = '';
    state.availableConsoles = consoleFolders;
    state.initialConsoleListItems = [];

    if (consoleFolders && consoleFolders.length > 0) {
        consoleFolders.forEach((folder) => {
            const initialLi = document.createElement('li');
            populateConsoleListItemContent(initialLi, folder);
            initialLi.addEventListener('click', () => selectInitialConsole(folder));
            dom.initialConsoleListUl.appendChild(initialLi);
            state.initialConsoleListItems.push(initialLi);

            const dropdownLi = document.createElement('li');
            dropdownLi.textContent = folder;
            dropdownLi.addEventListener('click', () => {
                dom.consoleSelectButton.textContent = folder;
                api.loadGamesFromConsole(folder);
                toggleConsoleMenu();
            });
            dom.consoleListUl.appendChild(dropdownLi);
        });

        if (state.isInitialSelectionActive) {
            state.initialConsoleListItemIndex = 0;
            updateInitialConsoleListSelection();
            state.initialConsoleListItems[0]?.focus();
        }
        dom.consoleSelectButton.disabled = false;
    } else {
        const noConsoleMsg = 'No console folders found';
        const li = document.createElement('li');
        li.textContent = noConsoleMsg;
        li.style.cursor = 'default';
        dom.initialConsoleListUl.appendChild(li.cloneNode(true));
        dom.consoleListUl.appendChild(li);
        dom.consoleSelectButton.disabled = true;
    }
}

function populateConsoleListItemContent(liElement, folderName) {
    const iconPath = `${state.paths.iconsBasePathUrl}/${folderName}.png`;
    const iconImg = document.createElement('img');
    iconImg.src = iconPath;
    iconImg.alt = folderName;
    iconImg.onerror = () => { iconImg.style.display = 'none'; };

    const iconWrapper = document.createElement('div');
    iconWrapper.className = 'console-icon-wrapper';
    iconWrapper.appendChild(iconImg);

    const nameSpan = document.createElement('span');
    nameSpan.className = 'console-list-name';
    nameSpan.textContent = folderName;

    liElement.appendChild(iconWrapper);
    liElement.appendChild(nameSpan);
    liElement.dataset.consoleName = folderName;

    if (NETPLAY_CONSOLES.includes(folderName)) {
        const netplayLabel = document.createElement('span');
        netplayLabel.className = 'console-netplay-label';
        netplayLabel.textContent = 'Netplay Available';
        liElement.appendChild(netplayLabel);
    }
}

export function selectInitialConsole(folderName) {
    if (!folderName) return;
    dom.consoleSelectButton.textContent = folderName;
    api.loadGamesFromConsole(folderName);
    hideInitialSelectionScreen();
    showNotification();
}

export function toggleConsoleMenu() {
    if (state.isInitialSelectionActive || state.isSupportPopupActive || state.isEmuAppRunning || state.isPlayDownloadMenuActive || state.isSaveManageMenuActive || state.isVersionMenuActive || state.isSavePromptActive || state.isTabMenuActive || state.isLobbyMenuActive || state.isVirtualKeyboardActive) return;
    state.isConsoleMenuActive = !state.isConsoleMenuActive;
    dom.consoleMenu.classList.toggle('hidden', !state.isConsoleMenuActive);
}

export function showInitialSelectionScreen() {
    dom.initialSelectionOverlay.classList.remove('hidden');
    document.body.classList.add('initial-selection-active');
    state.isInitialSelectionActive = true;
    state.navigationMode = 'initialConsoleList';
    if (state.initialConsoleListItems.length > 0) {
        state.initialConsoleListItemIndex = 0;
        updateInitialConsoleListSelection();
        state.initialConsoleListItems[0]?.focus();
    }
    if (!state.currentGames || state.currentGames.length === 0) {
        dom.gameScreenshotImg.src = state.paths.welcomeImageUrl;
        dom.gameScreenshotImg.alt = 'Welcome';
    }
}

export function hideInitialSelectionScreen() {
    dom.initialSelectionOverlay.classList.add('hidden');
    document.body.classList.remove('initial-selection-active');
    state.isInitialSelectionActive = false;
    state.navigationMode = 'gameList';
    if (state.currentGames.length > 0) {
        selectGame(state.selectedIndex !== -1 ? state.selectedIndex : 0, true, false);
        const focusIndex = state.selectedIndex !== -1 ? state.selectedIndex : 0;
        const target = dom.gameListUl.children[focusIndex] || dom.gameListUl.children[0];
        target?.focus();
    } else {
        updateGameImage(-1);
        dom.consoleSelectButton?.focus();
    }
}

export function updateInitialConsoleListSelection() {
    state.initialConsoleListItems.forEach((item, index) => {
        item.classList.toggle('selected', index === state.initialConsoleListItemIndex);
        if (index === state.initialConsoleListItemIndex) {
            item.focus();
            item.scrollIntoView({ block: 'nearest' });
        }
    });
}

export function openVersionMenu(gameGroup) {
    if (state.isInitialSelectionActive || state.isSupportPopupActive || state.isEmuAppRunning || state.isPlayDownloadMenuActive || state.isSaveManageMenuActive || state.isSavePromptActive || state.isLobbyMenuActive || state.isVirtualKeyboardActive) return;
    if (!gameGroup || !gameGroup.versions || gameGroup.versions.length <= 1) return;
    state.selectedGameGroup = gameGroup;
    dom.versionButtonsDiv.innerHTML = '';
    state.versionButtons = [];
    dom.versionMenuTitle.textContent = `Select Version: ${gameGroup.baseName}`;
        [...gameGroup.versions].reverse().forEach((version) => {
        const versionButton = document.createElement('button');
        versionButton.classList.add('version-button', 'menu-button');
        versionButton.textContent = version.title;
        versionButton.addEventListener('click', () => {
            closeVersionMenu();
            showPlayDownloadMenu(gameGroup, version);
        });
        dom.versionButtonsDiv.appendChild(versionButton);
        state.versionButtons.push(versionButton);
    });
    dom.versionMenuContainer.classList.remove('hidden');
    state.isVersionMenuActive = true;
    state.versionButtonIndex = 0;
    updateVersionButtonSelection();
    state.navigationMode = 'versionMenu';
    state.versionButtons[0]?.focus();
}

export function closeVersionMenu() {
    dom.versionMenuContainer.classList.add('hidden');
    state.isVersionMenuActive = false;
    state.navigationMode = 'gameList';
    if (!state.isInitialSelectionActive && dom.gameListUl.children[state.selectedIndex]) {
        dom.gameListUl.children[state.selectedIndex].focus();
    }
}

export function updateVersionButtonSelection() {
    state.versionButtons.forEach((button, index) => {
        button.classList.toggle('selected', index === state.versionButtonIndex);
        if (index === state.versionButtonIndex) button.focus();
    });
}

export function showPlayDownloadMenu(gameGroup, selectedVersion = null) {
    if (state.isInitialSelectionActive || state.isPlayDownloadMenuActive || state.isSaveManageMenuActive || state.isVersionMenuActive || state.isTabMenuActive || state.isSupportPopupActive || state.isEmuAppRunning || state.isSavePromptActive || state.isLobbyMenuActive || state.isVirtualKeyboardActive) return;
    const versionToUse = selectedVersion || gameGroup.versions[0];
    if (!versionToUse || !versionToUse.applicationPath) {
        alert("Error: Could not find valid game data for this selection.");
        return;
    }

    const buttonContainer = dom.playButton.parentElement;
    const isArcade = state.selectedConsoleName?.startsWith('Arcade');

    if (isArcade) {
        buttonContainer.insertBefore(dom.playOnlineButton, buttonContainer.firstChild);
        state.playDownloadMenuButtons = [dom.playOnlineButton, dom.playButton, dom.downloadOptionsButton];
    } else {
        buttonContainer.insertBefore(dom.playButton, buttonContainer.firstChild);
        state.playDownloadMenuButtons = [dom.playButton, dom.playOnlineButton, dom.downloadOptionsButton];
    }

    state.currentGamePathForPlayDownloadMenu = versionToUse.applicationPath;
    state.currentGameBaseNameForPlayDownloadMenu = gameGroup.baseName;
    state.selectedGameGroupForMenu = gameGroup;
    const isNetplaySupported = NETPLAY_CONSOLES.includes(state.selectedConsoleName);
    dom.playOnlineButton.disabled = !isNetplaySupported;
    dom.playOnlineButton.classList.toggle('disabled', !isNetplaySupported);
    dom.playDownloadMenuTitle.textContent = gameGroup.baseName;
    dom.playDownloadMenuContainer.classList.remove('hidden');
    state.isPlayDownloadMenuActive = true;
    state.playDownloadMenuIndex = 0;
    updatePlayDownloadMenuSelection();
    state.navigationMode = 'playDownloadMenu';
    state.playDownloadMenuButtons[0]?.focus();
}
export function closePlayDownloadMenu() {
    dom.playDownloadMenuContainer.classList.add('hidden');
    state.isPlayDownloadMenuActive = false;
    state.navigationMode = 'gameList';
    if (!state.isInitialSelectionActive && dom.gameListUl.children[state.selectedIndex]) {
        dom.gameListUl.children[state.selectedIndex].focus();
    }
}

export function updatePlayDownloadMenuSelection() {
    state.playDownloadMenuButtons.forEach((button, index) => {
        button.classList.toggle('selected', index === state.playDownloadMenuIndex);
        if (index === state.playDownloadMenuIndex) button.focus();
    });
}

export function showSaveManageMenu(gamePath, gameBaseName) {
    if (state.isInitialSelectionActive || state.isSaveManageMenuActive) return;
    closePlayDownloadMenu();
    state.currentGamePathForSaveMenu = gamePath;
    dom.saveManageMenuTitle.textContent = `Save Options: ${gameBaseName}`;
    dom.saveManageMenuContainer.classList.remove('hidden');
    state.isSaveManageMenuActive = true;
    state.saveManageMenuIndex = 0;
    updateSaveManageMenuSelection();
    state.navigationMode = 'saveManageMenu';
    state.saveManageMenuButtons[0]?.focus();
}

export function closeSaveManageMenu() {
    dom.saveManageMenuContainer.classList.add('hidden');
    state.isSaveManageMenuActive = false;
    state.navigationMode = 'gameList';
    if (!state.isInitialSelectionActive && dom.gameListUl.children[state.selectedIndex]) {
        dom.gameListUl.children[state.selectedIndex].focus();
    }
}

export function updateSaveManageMenuSelection() {
    state.saveManageMenuButtons.forEach((button, index) => {
        button.classList.toggle('selected', index === state.saveManageMenuIndex);
        if (index === state.saveManageMenuIndex) button.focus();
    });
}

export function showSavePrompt(tempPath, platform) {
    if (state.isInitialSelectionActive) return;
    state.gamePathToPotentiallySave = tempPath;
    state.platformForPotentialSave = platform;
    dom.savePromptContainer.classList.remove('hidden');
    state.isSavePromptActive = true;
    state.navigationMode = 'savePrompt';
    state.savePromptButtonIndex = 0;
    updateSavePromptButtonSelection();
    state.savePromptButtons[0]?.focus();
}

export function hideSavePrompt() {
    dom.savePromptContainer.classList.add('hidden');
    state.isSavePromptActive = false;
    state.gamePathToPotentiallySave = null;
    state.platformForPotentialSave = null;
    state.navigationMode = 'gameList';
    const focusTarget = dom.gameListUl.children[state.selectedIndex] || dom.gameListUl.children[0];
    if (!state.isInitialSelectionActive) focusTarget?.focus();
}

export function updateSavePromptButtonSelection() {
    state.savePromptButtons.forEach((button, index) => {
        button.classList.toggle('selected', index === state.savePromptButtonIndex);
        if (index === state.savePromptButtonIndex) button.focus();
    });
}

export async function openTabMenu() {
    if (state.isInitialSelectionActive || state.isEmuAppRunning || state.isSupportPopupActive || state.isPlayDownloadMenuActive || state.isSaveManageMenuActive || state.isVersionMenuActive || state.isSavePromptActive || state.isLobbyMenuActive || state.isVirtualKeyboardActive || state.isPowerMenuActive) return;
    dom.tabMenuContainer.classList.remove('hidden');
    state.isTabMenuActive = true;
    state.tabMenuButtonIndex = 0;
    updateTabMenuButtonSelection();
    state.navigationMode = 'tabMenu';
    dom.tabMenuButtonsDiv.style.display = 'flex';
    dom.tabConsoleListContainer.classList.add('hidden');
    state.isVirtualKeyboardActive = false;
    state.tabMenuButtons[0]?.focus();

    const currentConfig = await api.getRelayConfig();
    const foundIndex = state.relay.options.findIndex(opt => opt.server === currentConfig.server && opt.enabled === currentConfig.enabled);
    state.relay.currentIndex = (foundIndex !== -1) ? foundIndex : 0;
    state.relay.initialIndex = state.relay.currentIndex;
    updateRelayConfigDisplay();
}

export function closeTabMenu() {
    if (state.relay.currentIndex !== state.relay.initialIndex) {
        const newConfig = state.relay.options[state.relay.currentIndex];
        api.setRelayConfig(newConfig);
    }
    dom.tabMenuContainer.classList.add('hidden');
    state.isTabMenuActive = false;
    state.navigationMode = 'gameList';
    if (!state.isInitialSelectionActive && dom.gameListUl.children[state.selectedIndex]) {
        dom.gameListUl.children[state.selectedIndex].focus();
    } else if (state.currentGames.length > 0) {
        selectGame(0, true, false);
        dom.gameListUl.children[0]?.focus();
    }
}

export function updateTabMenuButtonSelection() {
    state.tabMenuButtons.forEach((button, index) => {
        button.classList.toggle('selected', index === state.tabMenuButtonIndex);
        if (index === state.tabMenuButtonIndex) button.focus();
    });
}

export function updateRelayConfigDisplay() {
    if (dom.relayConfigValue) {
        dom.relayConfigValue.textContent = state.relay.options[state.relay.currentIndex].label;
    }
}

export function showTabConsoleList() {
    dom.tabConsoleListContainer.classList.remove('hidden');
    dom.tabConsoleListUl.innerHTML = '';
    state.tabConsoleListItems = [];
    state.availableConsoles.forEach((folder) => {
        const li = document.createElement('li');
        populateConsoleListItemContent(li, folder);
        li.addEventListener('click', () => {
            dom.consoleSelectButton.textContent = folder;
            api.loadGamesFromConsole(folder);
            closeTabMenu();
        });
        dom.tabConsoleListUl.appendChild(li);
        state.tabConsoleListItems.push(li);
    });
    if (state.tabConsoleListItems.length > 0) {
        state.tabConsoleListItemIndex = 0;
        updateTabConsoleListSelection();
        state.navigationMode = 'tabConsoleList';
        dom.tabMenuButtonsDiv.style.display = 'none';
        state.tabConsoleListItems[0]?.focus();
    }
}

export function hideTabConsoleList() {
    dom.tabConsoleListContainer.classList.add('hidden');
    state.tabConsoleListItemIndex = -1;
    state.navigationMode = 'tabMenu';
    updateTabMenuButtonSelection();
    dom.tabMenuButtonsDiv.style.display = 'flex';
    state.tabMenuButtons[state.tabMenuButtonIndex]?.focus();
}

export function updateTabConsoleListSelection() {
    state.tabConsoleListItems.forEach((item, index) => {
        item.classList.toggle('selected', index === state.tabConsoleListItemIndex);
        if (index === state.tabConsoleListItemIndex) {
            item.focus();
            item.scrollIntoView({ block: 'nearest' });
        }
    });
}

export function createVirtualKeyboard() {
    const layout = [
        ['1', '2', '3', '4', '5', '6', '7', '8', '9', '0'],
        ['q', 'w', 'e', 'r', 't', 'y', 'u', 'i', 'o', 'p'],
        ['a', 's', 'd', 'f', 'g', 'h', 'j', 'k', 'l', '.'],
        ['Shift', 'z', 'x', 'c', 'v', 'b', 'n', 'm', '←'],
        [' ', 'Enter']
    ];
    dom.virtualKeyboardGrid.innerHTML = '';
    state.keyboardGrid = [];

    layout.forEach((row, rowIndex) => {
        const rowData = [];
        row.forEach((key, colIndex) => {
            const btn = document.createElement('button');
            btn.className = 'keyboard-button';
            btn.dataset.key = key;

            const keyInfo = { key, button: btn, row: rowIndex, col: colIndex, span: 1 };

            if (key === 'Shift') {
                btn.textContent = 'Shift';
                btn.classList.add('shift-key');
                keyInfo.span = 1;
            } else if (key === '←') {
                btn.textContent = '←';
                btn.classList.add('backspace-key');
                keyInfo.span = 2;
            } else if (key === ' ') {
                btn.textContent = 'Space';
                btn.classList.add('space-key');
                keyInfo.span = 7;
            } else if (key === 'Enter') {
                btn.textContent = 'Enter';
                btn.classList.add('enter-key');
                keyInfo.span = 3;
            } else {
                btn.textContent = key;
            }

            btn.addEventListener('click', () => handleKeyboardButtonClick(key));
            dom.virtualKeyboardGrid.appendChild(btn);
            rowData.push(keyInfo);
        });
        state.keyboardGrid.push(rowData);
    });
}

export function handleKeyboardButtonClick(key) {
    let currentInput = dom.virtualKeyboardInput.value;
    const maxLength = parseInt(dom.virtualKeyboardInput.getAttribute('maxlength'), 10);

    if (key === 'Enter') {
        if (state.onKeyboardConfirm) {
            state.onKeyboardConfirm(dom.virtualKeyboardInput.value);
        }
        closeVirtualKeyboard();
    } else if (key === 'Shift') {
        state.isShiftActive = !state.isShiftActive;
        state.keyboardGrid.flat().forEach(keyInfo => {
            if (keyInfo.key === 'Shift') {
                keyInfo.button.classList.toggle('active', state.isShiftActive);
            } else if (keyInfo.key.length === 1 && keyInfo.key.match(/[a-z]/i)) {
                keyInfo.button.textContent = state.isShiftActive ? keyInfo.key.toUpperCase() : keyInfo.key.toLowerCase();
            }
        });
    } else if (key === '←') {
        dom.virtualKeyboardInput.value = currentInput.slice(0, -1);
    } else if (key === ' ') {
        if (!maxLength || currentInput.length < maxLength) {
            dom.virtualKeyboardInput.value += ' ';
        }
    } else {
        if (!maxLength || currentInput.length < maxLength) {
            const charToAdd = state.isShiftActive ? key.toUpperCase() : key;
            dom.virtualKeyboardInput.value += charToAdd;
        }
    }
}

const onVirtualKeyboardInputKeydown = (event) => {
    if (event.key === 'Enter') {
        event.preventDefault(); 
        event.stopPropagation();
        if (state.onKeyboardConfirm) {
            state.onKeyboardConfirm(dom.virtualKeyboardInput.value);
        }
        closeVirtualKeyboard();
    }
};

export function openVirtualKeyboard(options = {}) {
    const { title = 'Input Text', placeholder = '...', initialValue = '', maxLength = null, onConfirm = () => {}, autoFocusInput = false } = options;
    if (state.isVirtualKeyboardActive) return;
    dom.virtualKeyboardTitle.textContent = title;
    dom.virtualKeyboardInput.placeholder = placeholder;
    dom.virtualKeyboardInput.value = initialValue;
    if (maxLength) {
        dom.virtualKeyboardInput.setAttribute('maxlength', maxLength);
    } else {
        dom.virtualKeyboardInput.removeAttribute('maxlength');
    }
    state.onKeyboardConfirm = onConfirm;
    dom.virtualKeyboardInput.readOnly = false;
    dom.virtualKeyboardInput.addEventListener('keydown', onVirtualKeyboardInputKeydown);
    createVirtualKeyboard();
    dom.virtualKeyboardModalContainer.classList.remove('hidden');
    state.isVirtualKeyboardActive = true;
    state.isShiftActive = false;
    state.navigationMode = 'virtualKeyboard';
    if (autoFocusInput) {
        dom.virtualKeyboardInput.focus();
    } else {
        state.keyboardButtonIndex = 0;
        updateKeyboardButtonSelection();
    }
}

export function closeVirtualKeyboard() {
    dom.virtualKeyboardModalContainer.classList.add('hidden');
    state.isVirtualKeyboardActive = false;
    state.onKeyboardConfirm = null;
    state.isShiftActive = false;
    dom.virtualKeyboardInput.readOnly = true;
    dom.virtualKeyboardInput.removeEventListener('keydown', onVirtualKeyboardInputKeydown);
    if(state.isTabMenuActive) {
        state.navigationMode = 'tabMenu';
        if (state.tabMenuButtons[state.tabMenuButtonIndex]) {
            state.tabMenuButtons[state.tabMenuButtonIndex].focus();
        }
    } else {
        state.navigationMode = 'gameList';
        if (dom.gameListUl.children[state.selectedIndex]) {
            dom.gameListUl.children[state.selectedIndex].focus();
        }
    }
}

export function updateKeyboardButtonSelection() {
    state.keyboardGrid.flat().forEach((keyInfo, index) => {
        keyInfo.button.classList.toggle('selected', index === state.keyboardButtonIndex);
        if (index === state.keyboardButtonIndex) {
            keyInfo.button.focus();
        }
    });
}

export function openPowerMenu() {
    closeTabMenu();
    dom.powerMenuContainer.classList.remove('hidden');
    state.isPowerMenuActive = true;
    state.navigationMode = 'powerMenu';
    state.powerMenuButtonIndex = 0;
    updatePowerMenuSelection();
}

export function closePowerMenu() {
    dom.powerMenuContainer.classList.add('hidden');
    state.isPowerMenuActive = false;
    state.navigationMode = 'gameList';
    if (!state.isInitialSelectionActive && dom.gameListUl.children[state.selectedIndex]) {
        dom.gameListUl.children[state.selectedIndex].focus();
    }
}

export function updatePowerMenuSelection() {
    state.powerMenuButtons.forEach((button, index) => {
        button.classList.toggle('selected', index === state.powerMenuButtonIndex);
        if (index === state.powerMenuButtonIndex) {
            button.focus();
        }
    });
}

export function showSupportPopup(customText = null) {
    if (state.isInitialSelectionActive) return;
    
    if (customText) {
        dom.supportPopupContainer.querySelector('#support-popup-subtitle').textContent = customText;
        dom.supportPopupContainer.querySelector('#support-popup-title').textContent = "Supporter Feature";
        dom.supportPopupContainer.querySelector('#support-popup-body').textContent = "You can get the updated app with all consoles and no popup.";
    } else {
        dom.supportPopupContainer.querySelector('#support-popup-title').textContent = "Thanks for using CloudBox";
        dom.supportPopupContainer.querySelector('#support-popup-subtitle').textContent = "Consider supporting the project in Patreon";
        dom.supportPopupContainer.querySelector('#support-popup-body').textContent = "You can get the updated app with more consoles and no popup";
    }

    dom.supportPopupContainer.classList.remove('hidden');
    state.isSupportPopupActive = true;
    state.navigationMode = 'supportPopup';
    state.supportPopupButtonIndex = 0;
    updateSupportPopupButtonSelection();
    state.supportPopupButtons[0]?.focus();
}

export function hideSupportPopup() {
    dom.supportPopupContainer.classList.add('hidden');
    state.isSupportPopupActive = false;
    state.navigationMode = 'gameList';
    if (!state.isInitialSelectionActive && dom.gameListUl.children[state.selectedIndex]) {
        dom.gameListUl.children[state.selectedIndex].focus();
    }
}

export function updateSupportPopupButtonSelection() {
    state.supportPopupButtons.forEach((button, index) => {
        button.classList.toggle('selected', index === state.supportPopupButtonIndex);
        if (index === state.supportPopupButtonIndex) button.focus();
    });
}

export function openLobbyMenu() {
    dom.lobbyMenuContainer.classList.remove('hidden');
    state.isLobbyMenuActive = true;
    state.navigationMode = 'lobbyMenu';
}

export function closeLobbyMenu() {
    dom.lobbyMenuContainer.classList.add('hidden');
    state.isLobbyMenuActive = false;
    state.lobbyGames = [];
    state.lobbySelectionIndex = -1;
    state.navigationMode = 'gameList';
    if (!state.isInitialSelectionActive && dom.gameListUl.children[state.selectedIndex]) {
        dom.gameListUl.children[state.selectedIndex].focus();
    }
}

export function renderLobbyList() {
    dom.lobbyListUl.innerHTML = '';
    if (state.lobbyGames.length === 0) {
        dom.lobbyListUl.innerHTML = '<li>No CloudBox games found in the online lobby.</li>';
        return;
    }
    state.lobbyGames.forEach((game, index) => {
        const li = document.createElement('li');
        li.dataset.index = index;
        li.textContent = `${game.username} is currently Playing ${game.game_name} on ${game.console_name}`;
        li.addEventListener('click', () => {
            closeLobbyMenu();
            api.triggerPlayNetplayClient(game);
        });
        dom.lobbyListUl.appendChild(li);
    });
}

export function updateLobbySelection() {
    const items = dom.lobbyListUl.children;
    for (let i = 0; i < items.length; i++) {
        items[i].classList.toggle('selected', i === state.lobbySelectionIndex);
        if (i === state.lobbySelectionIndex) {
            items[i].scrollIntoView({ block: 'nearest' });
        }
    }
}

function showNotification() {
    const data = state.notification.data;
    if (!data || !data.enabled) return;
    hideNotification(true);
    dom.dynamicNotificationTitle.textContent = data.title || '';
    dom.dynamicNotificationBody.textContent = data.bodytext || '';
    if (data.link && data.link.url && data.link.text) {
        dom.dynamicNotificationLink.href = data.link.url;
        dom.dynamicNotificationLink.textContent = data.link.text;
        dom.dynamicNotificationLink.classList.remove('hidden');
    } else {
        dom.dynamicNotificationLink.classList.add('hidden');
    }
    dom.dynamicNotification.classList.remove('hidden');
    requestAnimationFrame(() => {
        dom.dynamicNotification.classList.add('notification-visible');
    });
    const duration = (data.duration_seconds || 10) * 1000;
    state.notification.timeoutId = setTimeout(hideNotification, duration);
}

function hideNotification(instant = false) {
    if (state.notification.timeoutId) {
        clearTimeout(state.notification.timeoutId);
        state.notification.timeoutId = null;
    }
    if (instant) {
        dom.dynamicNotification.classList.add('hidden');
        dom.dynamicNotification.classList.remove('notification-visible');
    } else {
        dom.dynamicNotification.classList.remove('notification-visible');
        setTimeout(() => {
            dom.dynamicNotification.classList.add('hidden');
        }, 500);
    }
}

export async function initiateLaunchSequence(gamePath, launchType) {
    if (launchType === 'netplay') {
        await proceedWithNetplayLaunch(gamePath);
    } else {
        await proceedWithActualLaunch(gamePath);
    }
}

export async function proceedWithActualLaunch(gamePath) {
    if (!gamePath) return;
    prepareForOperation('play');
    dom.progressStatus.textContent = 'Preparing to launch...';
    try {
        const pid = await window.electronAPI.playGame(gamePath);
        state.currentEmuAppPid = pid || null;
    } catch (error) {
        showErrorInProgressBar(`Error: ${error.message || 'Launch failed'}`);
    }
}

export async function proceedWithNetplayLaunch(gamePath) {
    prepareForOperation('play-netplay-host');
    dom.progressStatus.textContent = 'Preparing to host online game...';
    try {
        const username = await api.getUsername();
        if (!username) {
            resetProgressOverlayState();
            openVirtualKeyboard({
                title: 'Set Online Username to Play',
                placeholder: 'Enter username...',
                maxLength: 12,
                autoFocusInput: true,
                onConfirm: async (newUsername) => {
                    if (newUsername.trim()) {
                        api.setUsername(newUsername.trim());
                        await proceedWithNetplayLaunch(gamePath);
                    }
                }
            });
            return;
        }
        const pid = await window.electronAPI.playNetplayHost(gamePath);
        state.currentEmuAppPid = pid || null;
    } catch (error) {
        showErrorInProgressBar(`Error: ${error.message || 'Netplay host failed'}`);
    }
}

export function prepareForOperation(operationType, gamePath = null) {
    resetProgressOverlayState();
    state.lastSuccessfulGamePath = null;
    state.currentOperationType = operationType;
    if (operationType === 'save-app' || operationType === 'save-external') {
        state.lastSuccessfulGamePath = gamePath;
    }
    dom.progressOverlay.classList.remove('hidden');
    dom.progressStatus.textContent = 'Preparing...';
    dom.progressBar.style.width = '0%';
}

export function handleProgressUpdate(progressData) {
    if (state.isSavePromptActive) return;
    if (progressData.type === 'prompt_save') {
        if (state.isEmuAppRunning || state.isSupportPopupActive || state.isPlayDownloadMenuActive || state.isSaveManageMenuActive || state.isVersionMenuActive || state.isTabMenuActive || state.isLobbyMenuActive || state.isVirtualKeyboardActive) return;
        showSavePrompt(progressData.temp_path, progressData.platform);
        return;
    }
    dom.progressBarContainer.classList.remove('hidden');
    dom.progressCompletionButtons.classList.add('hidden');
    dom.progressCancelHint.classList.remove('hidden');
    if (!dom.supportPopupContainer.classList.contains('hidden')) hideSupportPopup();
    dom.progressOverlay.classList.remove('hidden');
    dom.progressBar.classList?.remove('error');
    if (dom.progressBar.style) dom.progressBar.style.backgroundColor = '';
    dom.progressInfoText.textContent = '';
    const currentStage = progressData.stage;
    if (currentStage === 'download' && !state.isDownloading) startDownloadTimer();
    else if (currentStage !== 'download' && state.isDownloading) stopDownloadTimer();
    if ((currentStage === 'download' || currentStage === 'extract' || currentStage === 'save' || currentStage === 'move') && DISC_BASED_CONSOLES.includes(state.selectedConsoleName) && !state.isDownloading) {
        dom.progressInfoText.textContent = "Disc based games may take a longer time to process, please wait.";
    } else if (state.isDownloading) checkDownloadTime();
    
    if (currentStage === 'launch') {
        state.isEmuAppRunning = true;
    }

    if (progressData.type === 'error' && progressData.stage === 'launch_saved') {
        dom.progressStatus.textContent = `Error: ${progressData.message || 'Saved game corrupted, re-downloading...'}`;
    } else if (progressData.type === 'status') {
        dom.progressStatus.textContent = progressData.message || 'Processing...';
        if (['extract', 'save', 'move'].includes(currentStage)) {
            dom.progressBar.style.width = '100%';
        }
    } else if (progressData.type === 'progress') {
        if (!['extract', 'save', 'move'].includes(currentStage)) {
            const percent = progressData.percent !== undefined ? Math.max(0, Math.min(100, progressData.percent)) : 0;
            dom.progressBar.style.width = `${percent}%`;
            if (dom.progressPercentText) dom.progressPercentText.textContent = `${percent}%`;
            if (currentStage === 'download') dom.progressStatus.textContent = 'Downloading...';
        }
    } else if (progressData.type === 'error') {
        const supporterPrefix = 'SUPPORTER_LOCKED::';
        if (progressData.message && progressData.message.startsWith(supporterPrefix)) {
            const actualMessage = progressData.message.substring(supporterPrefix.length);
            resetProgressOverlayState();
            showSupportPopup(actualMessage);
            return;
        }
        showErrorInProgressBar(`Error: ${progressData.message || 'Unknown error'}`);
        stopDownloadTimer();
    }
}

export function handleEmuExit({ code, signal }) {
    stopDownloadTimer();
    if (dom.progressInfoText) dom.progressInfoText.textContent = '';
    const operationSucceeded = code === 0;
    const wasSaveOperation = ['save-app', 'save-external', 'move-to-saved'].includes(state.currentOperationType);
    if (operationSucceeded && wasSaveOperation) {
        dom.progressStatus.textContent = state.currentOperationType === 'move-to-saved' ? 'Game Saved to Collection!' : 'Game Saved Successfully!';
        dom.progressBarContainer.classList.add('hidden');
        dom.progressCancelHint.classList.add('hidden');
        dom.progressCompletionButtons.classList.remove('hidden');
        dom.progressAcceptButton.classList.remove('hidden');
        if (state.currentOperationType === 'save-app') {
            dom.progressPlayNowButton.classList.remove('hidden');
        } else {
            dom.progressPlayNowButton.classList.add('hidden');
        }
        if (state.postCompletionTimeoutId) clearTimeout(state.postCompletionTimeoutId);
        state.postCompletionTimeoutId = setTimeout(() => { resetProgressOverlayState(); state.lastSuccessfulGamePath = null; }, 30000);
        dom.progressAcceptButton.focus();
    } else {
        const isErrorShowing = dom.progressStatus?.textContent.startsWith('Error:');
        const hideDelay = (isErrorShowing && !operationSucceeded) ? 1500 : 100;
        if (state.postCompletionTimeoutId) clearTimeout(state.postCompletionTimeoutId);
        state.postCompletionTimeoutId = setTimeout(() => { resetProgressOverlayState(); state.lastSuccessfulGamePath = null; }, hideDelay);
    }
    state.isEmuAppRunning = false;
    state.currentEmuAppPid = null;
    api.scanLobbyAndShowAlert();
    const focusTarget = dom.gameListUl.children[state.selectedIndex] || (dom.letterButtons[state.letterButtonIndex] || null);
    if (!state.isInitialSelectionActive && !state.isSavePromptActive && !state.isSupportPopupActive && !state.isPlayDownloadMenuActive && !state.isSaveManageMenuActive && !state.isVersionMenuActive && !state.isTabMenuActive && !state.isVirtualKeyboardActive) {
        focusTarget?.focus();
    }
}

export function resetProgressOverlayState() {
    if (state.postCompletionTimeoutId) clearTimeout(state.postCompletionTimeoutId);
    state.postCompletionTimeoutId = null;
    dom.progressOverlay.classList.add('hidden');
    dom.progressBarContainer.classList.remove('hidden');
    dom.progressCompletionButtons.classList.add('hidden');
    dom.progressStatus.textContent = 'Starting...';
    dom.progressBar.style.width = '0%';
    dom.progressBar.style.backgroundColor = '';
    if(dom.progressPercentText) dom.progressPercentText.textContent = '';
    dom.progressInfoText.textContent = '';
    dom.progressCancelHint.classList.remove('hidden');
    state.currentOperationType = null;
}

export function showErrorInProgressBar(message) {
    if (state.isEmuAppRunning) state.isEmuAppRunning = false;
    if (state.currentEmuAppPid) state.currentEmuAppPid = null;
    dom.progressStatus.textContent = message;
    dom.progressBar.style.backgroundColor = 'red';
    dom.progressBar.style.width = '100%';
    if (state.postCompletionTimeoutId) clearTimeout(state.postCompletionTimeoutId);
    state.postCompletionTimeoutId = setTimeout(resetProgressOverlayState, 2000);
}

function checkDownloadTime() {
    if (state.isDownloading && state.downloadStartTime) {
        const elapsed = Date.now() - state.downloadStartTime;
        if (elapsed > DOWNLOAD_TIMEOUT_MS) {
            dom.progressInfoText.textContent = "If the server is under heavy load, downloads will take longer, you can wait or try again later";
        } else if (DISC_BASED_CONSOLES.includes(state.selectedConsoleName)) {
            dom.progressInfoText.textContent = "Disc based games may take a longer time to download and extract, please wait";
        }
    }
}

function startDownloadTimer() {
    if (state.downloadTimerId) clearInterval(state.downloadTimerId);
    state.downloadStartTime = Date.now();
    state.isDownloading = true;
    state.downloadTimerId = setInterval(checkDownloadTime, 1000);
}

function stopDownloadTimer() {
    if (state.downloadTimerId) clearInterval(state.downloadTimerId);
    state.downloadTimerId = null;
    state.isDownloading = false;
    state.downloadStartTime = null;
}

export function setupClickListeners() {
    dom.consoleSelectButton.addEventListener('click', toggleConsoleMenu);
    dom.changeConsoleButton.addEventListener('click', showTabConsoleList);
    dom.showLobbyButton.addEventListener('click', api.showOnlineLobby);
    dom.showFavoritesButton.addEventListener('click', () => {
        const favList = state.favoriteGames[state.selectedConsoleName] || [];
        const favoriteGames = state.allGames[state.selectedConsoleName]?.filter(g => favList.includes(g.baseName));
        state.currentGames = favoriteGames;
        state.isShowingFavorites = true;
        state.isShowingDownloaded = false;
        renderGameList(favoriteGames);
        closeTabMenu();
        if (favoriteGames.length > 0) selectGame(0, true, false);
    });
    dom.showDownloadedButton.addEventListener('click', api.showDownloadedGamesList);
    dom.searchGameButton.addEventListener('click', () => {
        closeTabMenu();
        openVirtualKeyboard({
            title: 'Search Game',
            placeholder: 'Type to search...',
            autoFocusInput: true,
            onConfirm: (searchValue) => {
                dom.gameSearchInput.value = searchValue;
                dom.gameSearchInput.dispatchEvent(new Event('input'));
            }
        });
    });

    dom.editUsernameButton.addEventListener('click', async () => {
        closeTabMenu();
        const currentUsername = await api.getUsername();
        openVirtualKeyboard({
            title: 'Edit Online Username',
            placeholder: 'Enter username...',
            initialValue: currentUsername || '',
            maxLength: 12,
            autoFocusInput: true,
            onConfirm: (newUsername) => {
                if (newUsername.trim()) {
                    api.setUsername(newUsername.trim());
                }
            }
        });
    });
    dom.powerOptionsButton.addEventListener('click', openPowerMenu);
    dom.quitAppButton.addEventListener('click', api.quitApp);
    dom.suspendButton.addEventListener('click', () => { closePowerMenu(); api.triggerSuspend(); });
    dom.shutdownButton.addEventListener('click', () => { closePowerMenu(); api.triggerShutdown(); });
    dom.restartButton.addEventListener('click', () => { closePowerMenu(); api.triggerRestart(); });
    dom.exitAppButton.addEventListener('click', () => { closePowerMenu(); api.quitApp(); });

    dom.playButton.addEventListener('click', () => {
        if (state.currentGamePathForPlayDownloadMenu) {
            closePlayDownloadMenu();
            api.triggerPlayGame(state.currentGamePathForPlayDownloadMenu);
        }
    });

    dom.playOnlineButton.addEventListener('click', async () => {
        if (dom.playOnlineButton.disabled || !state.currentGamePathForPlayDownloadMenu) return;
        const gamePath = state.currentGamePathForPlayDownloadMenu;
        closePlayDownloadMenu();
        await proceedWithNetplayLaunch(gamePath);
    });

    dom.downloadOptionsButton.addEventListener('click', () => {
        if (state.currentGamePathForPlayDownloadMenu && state.currentGameBaseNameForPlayDownloadMenu) {
            showSaveManageMenu(state.currentGamePathForPlayDownloadMenu, state.currentGameBaseNameForPlayDownloadMenu);
        }
    });

    dom.saveAppButton.addEventListener('click', () => {
        if (state.currentGamePathForSaveMenu) {
            closeSaveManageMenu();
            api.triggerSaveGameApp(state.currentGamePathForSaveMenu);
        }
    });
    dom.saveComputerButton.addEventListener('click', () => {
        if (state.currentGamePathForSaveMenu) {
            closeSaveManageMenu();
            api.triggerSaveGameExternal(state.currentGamePathForSaveMenu);
        }
    });
    dom.manageSavedButton.addEventListener('click', () => {
        closeSaveManageMenu();
        api.manageSavedGames();
    });
    dom.addToQueueButton.addEventListener('click', api.addToDownloadQueue);

    dom.supportPopupCloseButton.addEventListener('click', () => {
        const game = state.gameToLaunchAfterPopup;
        state.gameToLaunchAfterPopup = null;
        hideSupportPopup();
        if (game && game.path) {
            if (game.type === 'netplay') proceedWithNetplayLaunch(game.path);
            else proceedWithActualLaunch(game.path);
        }
    });
    dom.supportPopupPatreonButton.addEventListener('click', () => {
        if (dom.buyMeABeerLink?.href) window.electronAPI.openExternalLink(dom.buyMeABeerLink.href);
        state.gameToLaunchAfterPopup = null;
        hideSupportPopup();
    });

    dom.progressAcceptButton.addEventListener('click', () => {
        resetProgressOverlayState();
        state.lastSuccessfulGamePath = null;
        if (!state.isInitialSelectionActive && dom.gameListUl.children[state.selectedIndex]) {
            dom.gameListUl.children[state.selectedIndex].focus();
        }
    });
    dom.progressPlayNowButton.addEventListener('click', () => {
        const gamePath = state.lastSuccessfulGamePath;
        resetProgressOverlayState();
        state.lastSuccessfulGamePath = null;
        if (gamePath) api.triggerPlayGame(gamePath);
    });

    dom.savePromptYesButton.addEventListener('click', () => api.savePromptResponse('yes'));
    dom.savePromptNoButton.addEventListener('click', () => api.savePromptResponse('no'));

    dom.gameSearchInput.addEventListener('input', () => {
        const searchTerm = dom.gameSearchInput.value.toLowerCase().trim();
        state.isShowingFavorites = false;
        state.isShowingDownloaded = false;
        const allConsoleGames = state.allGames[state.selectedConsoleName] || [];
        state.currentGames = searchTerm
            ? allConsoleGames.filter(g => {
                let displayName = g.baseName;
                if (state.selectedConsoleName.startsWith('Arcade') && state.arcadeTitles && state.arcadeTitles[g.baseName]) {
                    displayName = state.arcadeTitles[g.baseName];
                }
                return displayName?.toLowerCase().includes(searchTerm);
            })
            : [...allConsoleGames];
        renderGameList(state.currentGames);
        if (state.currentGames.length > 0) {
            selectGame(0, true, false);
        } else {
            state.selectedIndex = -1;
            updateGameImage(-1);
            setActiveLetterButton(null);
        }
    });

    document.addEventListener('click', (event) => {
        if (state.isConsoleMenuActive && !dom.consoleSelectButton.contains(event.target) && !dom.consoleMenu.contains(event.target)) {
            toggleConsoleMenu();
        }
    });

    dom.letterButtons.forEach(button => {
        button.addEventListener('click', () => {
            const letter = button.dataset.letter;
            const foundIndex = state.currentGames.findIndex(g => {
                let title = g.baseName.toUpperCase();
                if (state.selectedConsoleName.startsWith('Arcade') && state.arcadeTitles && state.arcadeTitles[g.baseName]) {
                    title = state.arcadeTitles[g.baseName].toUpperCase();
                }
                return letter === '#' ? !/^[A-Z]/.test(title) : title.startsWith(letter);
            });
            if (foundIndex !== -1) selectGame(foundIndex, false, true);
            setActiveLetterButton(button);
            state.navigationMode = 'gameList';
            dom.letterJumpColumn.classList.remove('column-focused');
            if (state.selectedIndex >= 0 && dom.gameListUl.children[state.selectedIndex]) {
                dom.gameListUl.children[state.selectedIndex].focus();
            }
        });
    });

    dom.buyMeABeerLink?.addEventListener('click', (e) => { e.preventDefault(); window.electronAPI.openExternalLink(e.currentTarget.href); });
    dom.youtubeLinkBottom?.addEventListener('click', (e) => { e.preventDefault(); window.electronAPI.openExternalLink(e.currentTarget.href); });
    dom.dynamicNotificationLink?.addEventListener('click', (e) => { e.preventDefault(); window.electronAPI.openExternalLink(e.currentTarget.href); });
}